#!/usr/bin/perl

# rgape2id3v2.pl
#
# by Paul Rogers [paul.rogers at flumps.org] (c) 2024
#
# Provided as is - no warranty should be expected. Use at your own risk.
# Distributed under the GNU General Public License v3
#
# To be called from ReplayGain app after gain processing to copy the APE tags to ID3v2.
# It will overwrite any existing replay gain tags in ID3v2 tags with information in the APE tags,
# if all or partial APE information exists.
#
# Expects a single file that will have the APE replay gain tags read and copied to ID3v2.
#
# Requires:
#    MP3::Tag (installable from CPAN)
#    Audio::ApeTag (manually installable from https://www.flumps.org/mp3z/tagging/perl-audio-ape.zip)

use strict;
use warnings;

use Audio::ApeTag;
use MP3::Tag;

my $gsID3V2FrameRG = "TXXX";
my $gsID3V2FrameRGEncoding = "encoding";
my $gsID3V2FrameRGEncodingVal = "1";
my $gsID3V2FrameRGDescr = "Description";
my $gsID3V2FrameRGText = "Text";
my $gsTagRGGainAlbum = "REPLAYGAIN_ALBUM_GAIN";
my $gsTagRGPeakAlbum = "REPLAYGAIN_ALBUM_PEAK";
my $gsTagRGGainTrack = "REPLAYGAIN_TRACK_GAIN";
my $gsTagRGPeakTrack = "REPLAYGAIN_TRACK_PEAK";

my $sFile = $ARGV[0];

&sCopyRGTagsAPE2ID3v2( $sFile );

sub sCopyRGTagsAPE2ID3v2
{
	my $sFile = $_[0];

	if ( ( $sFile eq "" ) || ( !( defined( $sFile ) ) ) )
	{
		die "ERROR! Filename to copy APE gain tags to ID3v2 tags must NOT be blank.";
	}
	elsif ( !( -e $sFile ) )
	{
		die "ERROR! File to copy APE gain tags to ID3v2 tags does NOT exists / inaccessible: ".$sFile."\nError: .".$!;
	}
	elsif ( !( -w $sFile ) )
	{
		die "ERROR! Cannot write to file: ".$sFile."\nError: ".$!;
	}

	my $oAPE = Audio::ApeTag->new( $sFile );

	if ( $oAPE->has_tag )
	{
		my $sAlbumGain = "";
		my $sAlbumPeak = "";
		my $sTrackGain = "";
		my $sTrackPeak = "";

		my %hAPETags = %{ $oAPE->fields };

		foreach my $sKey ( keys %hAPETags )
		{
			my $raValues = $hAPETags{ $sKey };
			my $sAPEVal = @$raValues[ 0 ];

			if ( $sKey eq $gsTagRGGainAlbum ) { $sAlbumGain = $sAPEVal; }
			elsif ( $sKey eq $gsTagRGPeakAlbum ) { $sAlbumPeak = $sAPEVal; }
			elsif ( $sKey eq $gsTagRGGainTrack ) { $sTrackGain = $sAPEVal; }
			elsif ( $sKey eq $gsTagRGPeakTrack ) { $sTrackPeak = $sAPEVal; }
		}

		if ( ( $sAlbumGain eq "" ) && ( $sAlbumPeak eq "" ) && ( $sTrackGain eq "" ) && ( $sTrackPeak eq "" ) )
		{
			die "ERROR! All four (4) APE album gain tags are not present. Are we being called from the ReplayGain app?"
		}

		my $oMP3 = MP3::Tag->new( $sFile );
		my $oMP3ID3v2;
		$oMP3->get_tags();

		if ( exists $oMP3->{ID3v2} )
		{
			$oMP3ID3v2 = $oMP3->{ID3v2};

			my $rhFrames = $oMP3ID3v2->get_frame_ids();

			# Must loop through all frames as multiple TXXX frames are stored as TXXX%% where %% is a two digit identifier
			foreach my $sNameFrame ( keys %$rhFrames )
			{
				if ( $sNameFrame =~ m/^$gsID3V2FrameRG/ )
				{
					$oMP3ID3v2->remove_frame( $sNameFrame );
					# You must write the tag to actually remove the frame - feels like MP3::Tag should remove it in memory without a need to update the file (expensive!).
					$oMP3ID3v2->write_tag();
				}
			}

			my $sNameFrameAdd;

			if ( $sAlbumGain ne "" )
			{
				$sNameFrameAdd = $oMP3ID3v2->add_frame( $gsID3V2FrameRG, $gsID3V2FrameRGEncodingVal, lc( $gsTagRGGainAlbum ), $sAlbumGain );
			}
			if ( $sAlbumPeak ne "" )
			{
				$sNameFrameAdd = $oMP3ID3v2->add_frame( $gsID3V2FrameRG, $gsID3V2FrameRGEncodingVal, lc( $gsTagRGPeakAlbum ), $sAlbumPeak );
			}
			if ( $sTrackGain ne "" )
			{
				$sNameFrameAdd = $oMP3ID3v2->add_frame( $gsID3V2FrameRG, $gsID3V2FrameRGEncodingVal, lc( $gsTagRGGainTrack ), $sTrackGain );
			}
			if ( $sTrackPeak ne "" )
			{
				$sNameFrameAdd = $oMP3ID3v2->add_frame( $gsID3V2FrameRG, $gsID3V2FrameRGEncodingVal, lc( $gsTagRGPeakTrack ), $sTrackPeak );
			}

			$oMP3ID3v2->write_tag();
		}
		else
		{
			die "ERROR! ID3v2 tags must exist in the file already: ".$sFile;
		}
	}
}